/*
 * Copyright (C) 2004-2009, 2011  Internet Systems Consortium, Inc. ("ISC")
 * Copyright (C) 2000-2003  Internet Software Consortium.
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND ISC DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS.  IN NO EVENT SHALL ISC BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * Principal Author: Brian Wellington
 * $Id: opensslrsa_link.c,v 1.39 2011-01-11 23:47:13 tbox Exp $
 */
#define R_PKCS11

#ifdef OPENSSL
#include <config.h>

#ifndef USE_EVP
#if !defined(HAVE_EVP_SHA256) || !defined(HAVE_EVP_SHA512)
#define USE_EVP 0
#else
#define USE_EVP 1
#endif
#endif

#ifdef R_PKCS11
#define USE_EVP 0
#endif

#include <isc/entropy.h>
#include <isc/md5.h>
#include <isc/sha1.h>
#include <isc/sha2.h>
#include <isc/mem.h>
#include <isc/string.h>
#include <isc/util.h>

#include <dst/result.h>

#include "dst_internal.h"
#include "dst_openssl.h"
#include "dst_parse.h"

#include <openssl/err.h>
#include <openssl/objects.h>
#include <openssl/rsa.h>
#if OPENSSL_VERSION_NUMBER > 0x00908000L
#include <openssl/bn.h>
#endif
#include <openssl/engine.h>

/*
 * We don't use configure for windows so enforce the OpenSSL version
 * here.  Unlike with configure we don't support overriding this test.
 */
#ifdef WIN32
#if !((OPENSSL_VERSION_NUMBER >= 0x009070cfL && \
       OPENSSL_VERSION_NUMBER < 0x00908000L) || \
      OPENSSL_VERSION_NUMBER >= 0x0090804fL)
#error Please upgrade OpenSSL to 0.9.8d/0.9.7l or greater.
#endif
#endif


	/*
	 * XXXMPA  Temporarily disable RSA_BLINDING as it requires
	 * good quality random data that cannot currently be guaranteed.
	 * XXXMPA  Find which versions of openssl use pseudo random data
	 * and set RSA_FLAG_BLINDING for those.
	 */

#if 0
#if OPENSSL_VERSION_NUMBER < 0x0090601fL
#define SET_FLAGS(rsa) \
	do { \
	(rsa)->flags &= ~(RSA_FLAG_CACHE_PUBLIC | RSA_FLAG_CACHE_PRIVATE); \
	(rsa)->flags |= RSA_FLAG_BLINDING; \
	} while (0)
#else
#define SET_FLAGS(rsa) \
	do { \
		(rsa)->flags |= RSA_FLAG_BLINDING; \
	} while (0)
#endif
#endif

#if OPENSSL_VERSION_NUMBER < 0x0090601fL
#define SET_FLAGS(rsa) \
	do { \
	(rsa)->flags &= ~(RSA_FLAG_CACHE_PUBLIC | RSA_FLAG_CACHE_PRIVATE); \
	(rsa)->flags &= ~RSA_FLAG_BLINDING; \
	} while (0)
#elif defined(RSA_FLAG_NO_BLINDING)
#define SET_FLAGS(rsa) \
	do { \
		(rsa)->flags &= ~RSA_FLAG_BLINDING; \
		(rsa)->flags |= RSA_FLAG_NO_BLINDING; \
	} while (0)
#else
#define SET_FLAGS(rsa) \
	do { \
		(rsa)->flags &= ~RSA_FLAG_BLINDING; \
	} while (0)
#endif

#define DST_RET(a) {ret = a; goto err;}

#ifdef R_PKCS11
#include <dns/name.h>
#define MAXPATHLEN 512
static int use_pkcs11=0; /*off by default. on if PKCS11_LIBRARY_PATH defined*/
/* 
 * Caution: since this loads a 3rd party library and initializes it, 
 * it may reset signal() handling and the like. So pkcs11_initlib must 
 * be called in main() before anything else. 
 * Will check for PKCS11_LIBRARY_PATH lib and set.
 */
int pkcs11_initlib(char *pin);
static int pkcs11_RSA_sign(dst_key_t *key,int type,unsigned char *msg,unsigned int mlen,unsigned char *sign,unsigned int *slen);
static int pkcs11_parse(dst_key_t *key,isc_lex_t *lexer,unsigned char *modulus,int *modulus_len,unsigned char *pubexponent,int *pubexponent_len);
static void pkcs11_free(void *cb);
#define PKCS11_MAGIC 0x5A62
static int ispkcs11(const dst_key_t *key);
#endif /* R_PKCS11 */


static isc_result_t opensslrsa_todns(const dst_key_t *key, isc_buffer_t *data);

static isc_result_t
opensslrsa_createctx(dst_key_t *key, dst_context_t *dctx) {
#if USE_EVP
	EVP_MD_CTX *evp_md_ctx;
	const EVP_MD *type = NULL;
#endif

	UNUSED(key);
	REQUIRE(dctx->key->key_alg == DST_ALG_RSAMD5 ||
		dctx->key->key_alg == DST_ALG_RSASHA1 ||
		dctx->key->key_alg == DST_ALG_NSEC3RSASHA1 ||
		dctx->key->key_alg == DST_ALG_RSASHA256 ||
		dctx->key->key_alg == DST_ALG_RSASHA512);

#if USE_EVP
	evp_md_ctx = EVP_MD_CTX_create();
	if (evp_md_ctx == NULL)
		return (ISC_R_NOMEMORY);

	switch (dctx->key->key_alg) {
	case DST_ALG_RSAMD5:
		type = EVP_md5();	/* MD5 + RSA */
		break;
	case DST_ALG_RSASHA1:
	case DST_ALG_NSEC3RSASHA1:
		type = EVP_sha1();	/* SHA1 + RSA */
		break;
#ifdef HAVE_EVP_SHA256
	case DST_ALG_RSASHA256:
		type = EVP_sha256();	/* SHA256 + RSA */
		break;
#endif
#ifdef HAVE_EVP_SHA512
	case DST_ALG_RSASHA512:
		type = EVP_sha512();
		break;
#endif
	default:
		INSIST(0);
	}

	if (!EVP_DigestInit_ex(evp_md_ctx, type, NULL)) {
		EVP_MD_CTX_destroy(evp_md_ctx);
		return (ISC_R_FAILURE);
	}
	dctx->ctxdata.evp_md_ctx = evp_md_ctx;
#else
	switch (dctx->key->key_alg) {
	case DST_ALG_RSAMD5:
		{
			isc_md5_t *md5ctx;

			md5ctx = isc_mem_get(dctx->mctx, sizeof(isc_md5_t));
			if (md5ctx == NULL)
				return (ISC_R_NOMEMORY);
			isc_md5_init(md5ctx);
			dctx->ctxdata.md5ctx = md5ctx;
		}
		break;
	case DST_ALG_RSASHA1:
	case DST_ALG_NSEC3RSASHA1:
		{
			isc_sha1_t *sha1ctx;

			sha1ctx = isc_mem_get(dctx->mctx, sizeof(isc_sha1_t));
			if (sha1ctx == NULL)
				return (ISC_R_NOMEMORY);
			isc_sha1_init(sha1ctx);
			dctx->ctxdata.sha1ctx = sha1ctx;
		}
		break;
	case DST_ALG_RSASHA256:
		{
			isc_sha256_t *sha256ctx;

			sha256ctx = isc_mem_get(dctx->mctx,
						sizeof(isc_sha256_t));
			if (sha256ctx == NULL)
				return (ISC_R_NOMEMORY);
			isc_sha256_init(sha256ctx);
			dctx->ctxdata.sha256ctx = sha256ctx;
		}
		break;
	case DST_ALG_RSASHA512:
		{
			isc_sha512_t *sha512ctx;

			sha512ctx = isc_mem_get(dctx->mctx,
						sizeof(isc_sha512_t));
			if (sha512ctx == NULL)
				return (ISC_R_NOMEMORY);
			isc_sha512_init(sha512ctx);
			dctx->ctxdata.sha512ctx = sha512ctx;
		}
		break;
	default:
		INSIST(0);
	}
#endif

	return (ISC_R_SUCCESS);
}

static void
opensslrsa_destroyctx(dst_context_t *dctx) {
#if USE_EVP
	EVP_MD_CTX *evp_md_ctx = dctx->ctxdata.evp_md_ctx;
#endif

	REQUIRE(dctx->key->key_alg == DST_ALG_RSAMD5 ||
		dctx->key->key_alg == DST_ALG_RSASHA1 ||
		dctx->key->key_alg == DST_ALG_NSEC3RSASHA1 ||
		dctx->key->key_alg == DST_ALG_RSASHA256 ||
		dctx->key->key_alg == DST_ALG_RSASHA512);

#if USE_EVP
	if (evp_md_ctx != NULL) {
		EVP_MD_CTX_destroy(evp_md_ctx);
		dctx->ctxdata.evp_md_ctx = NULL;
	}
#else
	switch (dctx->key->key_alg) {
	case DST_ALG_RSAMD5:
		{
			isc_md5_t *md5ctx = dctx->ctxdata.md5ctx;

			if (md5ctx != NULL) {
				isc_md5_invalidate(md5ctx);
				isc_mem_put(dctx->mctx, md5ctx,
					    sizeof(isc_md5_t));
				dctx->ctxdata.md5ctx = NULL;
			}
		}
		break;
	case DST_ALG_RSASHA1:
	case DST_ALG_NSEC3RSASHA1:
		{
			isc_sha1_t *sha1ctx = dctx->ctxdata.sha1ctx;

			if (sha1ctx != NULL) {
				isc_sha1_invalidate(sha1ctx);
				isc_mem_put(dctx->mctx, sha1ctx,
					    sizeof(isc_sha1_t));
				dctx->ctxdata.sha1ctx = NULL;
			}
		}
		break;
	case DST_ALG_RSASHA256:
		{
			isc_sha256_t *sha256ctx = dctx->ctxdata.sha256ctx;

			if (sha256ctx != NULL) {
				isc_sha256_invalidate(sha256ctx);
				isc_mem_put(dctx->mctx, sha256ctx,
					    sizeof(isc_sha256_t));
				dctx->ctxdata.sha256ctx = NULL;
			}
		}
		break;
	case DST_ALG_RSASHA512:
		{
			isc_sha512_t *sha512ctx = dctx->ctxdata.sha512ctx;

			if (sha512ctx != NULL) {
				isc_sha512_invalidate(sha512ctx);
				isc_mem_put(dctx->mctx, sha512ctx,
					    sizeof(isc_sha512_t));
				dctx->ctxdata.sha512ctx = NULL;
			}
		}
		break;
	default:
		INSIST(0);
	}
#endif
}

static isc_result_t
opensslrsa_adddata(dst_context_t *dctx, const isc_region_t *data) {
#if USE_EVP
	EVP_MD_CTX *evp_md_ctx = dctx->ctxdata.evp_md_ctx;
#endif

	REQUIRE(dctx->key->key_alg == DST_ALG_RSAMD5 ||
		dctx->key->key_alg == DST_ALG_RSASHA1 ||
		dctx->key->key_alg == DST_ALG_NSEC3RSASHA1 ||
		dctx->key->key_alg == DST_ALG_RSASHA256 ||
		dctx->key->key_alg == DST_ALG_RSASHA512);

#if USE_EVP
	if (!EVP_DigestUpdate(evp_md_ctx, data->base, data->length)) {
		return (ISC_R_FAILURE);
	}
#else
	switch (dctx->key->key_alg) {
	case DST_ALG_RSAMD5:
		{
			isc_md5_t *md5ctx = dctx->ctxdata.md5ctx;

			isc_md5_update(md5ctx, data->base, data->length);
		}
		break;
	case DST_ALG_RSASHA1:
	case DST_ALG_NSEC3RSASHA1:
		{
			isc_sha1_t *sha1ctx = dctx->ctxdata.sha1ctx;

			isc_sha1_update(sha1ctx, data->base, data->length);
		}
		break;
	case DST_ALG_RSASHA256:
		{
			isc_sha256_t *sha256ctx = dctx->ctxdata.sha256ctx;

			isc_sha256_update(sha256ctx, data->base, data->length);
		}
		break;
	case DST_ALG_RSASHA512:
		{
			isc_sha512_t *sha512ctx = dctx->ctxdata.sha512ctx;

			isc_sha512_update(sha512ctx, data->base, data->length);
		}
		break;
	default:
		INSIST(0);
	}
#endif
	return (ISC_R_SUCCESS);
}

#if ! USE_EVP && OPENSSL_VERSION_NUMBER < 0x00908000L
/*
 * Digest prefixes from RFC 5702.
 */
static unsigned char sha256_prefix[] =
	 { 0x30, 0x31, 0x30, 0x0d, 0x06, 0x09, 0x60, 0x86, 0x48,
	   0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x04, 0x20};
static unsigned char sha512_prefix[] =
	 { 0x30, 0x51, 0x30, 0x0d, 0x06, 0x09, 0x60, 0x86, 0x48,
	   0x01, 0x65, 0x03, 0x04, 0x02, 0x03, 0x05, 0x00, 0x04, 0x40};
#define PREFIXLEN sizeof(sha512_prefix)
#else
#define PREFIXLEN 0
#endif

static isc_result_t
opensslrsa_sign(dst_context_t *dctx, isc_buffer_t *sig) {
	dst_key_t *key = dctx->key;
	isc_region_t r;
	unsigned int siglen = 0;
#if USE_EVP
	EVP_MD_CTX *evp_md_ctx = dctx->ctxdata.evp_md_ctx;
	EVP_PKEY *pkey = key->keydata.pkey;
#else
	RSA *rsa = key->keydata.rsa;
	/* note: ISC_SHA512_DIGESTLENGTH >= ISC_*_DIGESTLENGTH */
	unsigned char digest[PREFIXLEN + ISC_SHA512_DIGESTLENGTH];
	int status;
	int type = 0;
	unsigned int digestlen = 0;
	char *message;
	unsigned long err;
	const char* file;
	int line;
#if OPENSSL_VERSION_NUMBER < 0x00908000L
	unsigned int prefixlen = 0;
	const unsigned char *prefix = NULL;
#endif
#endif

	REQUIRE(dctx->key->key_alg == DST_ALG_RSAMD5 ||
		dctx->key->key_alg == DST_ALG_RSASHA1 ||
		dctx->key->key_alg == DST_ALG_NSEC3RSASHA1 ||
		dctx->key->key_alg == DST_ALG_RSASHA256 ||
		dctx->key->key_alg == DST_ALG_RSASHA512);

	isc_buffer_availableregion(sig, &r);

#if USE_EVP
	if (r.length < (unsigned int) EVP_PKEY_size(pkey))
		return (ISC_R_NOSPACE);

	if (!EVP_SignFinal(evp_md_ctx, r.base, &siglen, pkey)) {
		return (ISC_R_FAILURE);
	}
#else
	if (r.length < (unsigned int) RSA_size(rsa))
		return (ISC_R_NOSPACE);

	switch (dctx->key->key_alg) {
	case DST_ALG_RSAMD5:
		{
			isc_md5_t *md5ctx = dctx->ctxdata.md5ctx;

			isc_md5_final(md5ctx, digest);
			type = NID_md5;
			digestlen = ISC_MD5_DIGESTLENGTH;
		}
		break;
	case DST_ALG_RSASHA1:
	case DST_ALG_NSEC3RSASHA1:
		{
			isc_sha1_t *sha1ctx = dctx->ctxdata.sha1ctx;

			isc_sha1_final(sha1ctx, digest);
			type = NID_sha1;
			digestlen = ISC_SHA1_DIGESTLENGTH;
		}
		break;
	case DST_ALG_RSASHA256:
		{
			isc_sha256_t *sha256ctx = dctx->ctxdata.sha256ctx;

			isc_sha256_final(digest, sha256ctx);
			digestlen = ISC_SHA256_DIGESTLENGTH;
#if OPENSSL_VERSION_NUMBER < 0x00908000L
			prefix = sha256_prefix;
			prefixlen = sizeof(sha256_prefix);
#else
			type = NID_sha256;
#endif
		}
		break;
	case DST_ALG_RSASHA512:
		{
			isc_sha512_t *sha512ctx = dctx->ctxdata.sha512ctx;

			isc_sha512_final(digest, sha512ctx);
			digestlen = ISC_SHA512_DIGESTLENGTH;
#if OPENSSL_VERSION_NUMBER < 0x00908000L
			prefix = sha512_prefix;
			prefixlen = sizeof(sha512_prefix);
#else
			type = NID_sha512;
#endif
		}
		break;
	default:
		INSIST(0);
	}

#if OPENSSL_VERSION_NUMBER < 0x00908000L
	switch (dctx->key->key_alg) {
	case DST_ALG_RSAMD5:
	case DST_ALG_RSASHA1:
	case DST_ALG_NSEC3RSASHA1:
		INSIST(type != 0);
		status = RSA_sign(type, digest, digestlen, r.base,
				  &siglen, rsa);
		break;

	case DST_ALG_RSASHA256:
	case DST_ALG_RSASHA512:
		INSIST(prefix != NULL);
		INSIST(prefixlen != 0);
		INSIST(prefixlen + digestlen <= sizeof(digest));

		memmove(digest + prefixlen, digest, digestlen);
		memcpy(digest, prefix, prefixlen);
		status = RSA_private_encrypt(digestlen + prefixlen,
					     digest, r.base, rsa,
					     RSA_PKCS1_PADDING);
		if (status < 0)
			status = 0;
		else
			siglen = status;
		break;

	default:
		INSIST(0);
	}
#else
	INSIST(type != 0);
#ifdef R_PKCS11
        if(ispkcs11(key)) {
          siglen = r.length; /* pass size of buffer to signer */
          status = pkcs11_RSA_sign(key,type,digest,digestlen,r.base,&siglen);
          if(status == 0) status = 1; else status = 0;
        } else {
	  status = RSA_sign(type, digest, digestlen, r.base, &siglen, rsa);
	}
#else
	status = RSA_sign(type, digest, digestlen, r.base, &siglen, rsa);
#endif /* R_PKCS11 */

#endif
	if (status == 0) {
		err = ERR_peek_error_line(&file, &line);
		if (err != 0U) {
			message = ERR_error_string(err, NULL);
		}
		return (dst__openssl_toresult(DST_R_OPENSSLFAILURE));
	}
#endif

	isc_buffer_add(sig, siglen);

	return (ISC_R_SUCCESS);
}

static isc_result_t
opensslrsa_verify(dst_context_t *dctx, const isc_region_t *sig) {
	dst_key_t *key = dctx->key;
	int status = 0;
#if USE_EVP
	EVP_MD_CTX *evp_md_ctx = dctx->ctxdata.evp_md_ctx;
	EVP_PKEY *pkey = key->keydata.pkey;
#else
	/* note: ISC_SHA512_DIGESTLENGTH >= ISC_*_DIGESTLENGTH */
	unsigned char digest[ISC_SHA512_DIGESTLENGTH];
	int type = 0;
	unsigned int digestlen = 0;
	RSA *rsa = key->keydata.rsa;
#if OPENSSL_VERSION_NUMBER < 0x00908000L
	unsigned int prefixlen = 0;
	const unsigned char *prefix = NULL;
#endif
#endif

	REQUIRE(dctx->key->key_alg == DST_ALG_RSAMD5 ||
		dctx->key->key_alg == DST_ALG_RSASHA1 ||
		dctx->key->key_alg == DST_ALG_NSEC3RSASHA1 ||
		dctx->key->key_alg == DST_ALG_RSASHA256 ||
		dctx->key->key_alg == DST_ALG_RSASHA512);

#if USE_EVP
	status = EVP_VerifyFinal(evp_md_ctx, sig->base, sig->length, pkey);
#else
	switch (dctx->key->key_alg) {
	case DST_ALG_RSAMD5:
		{
			isc_md5_t *md5ctx = dctx->ctxdata.md5ctx;

			isc_md5_final(md5ctx, digest);
			type = NID_md5;
			digestlen = ISC_MD5_DIGESTLENGTH;
		}
		break;
	case DST_ALG_RSASHA1:
	case DST_ALG_NSEC3RSASHA1:
		{
			isc_sha1_t *sha1ctx = dctx->ctxdata.sha1ctx;

			isc_sha1_final(sha1ctx, digest);
			type = NID_sha1;
			digestlen = ISC_SHA1_DIGESTLENGTH;
		}
		break;
	case DST_ALG_RSASHA256:
		{
			isc_sha256_t *sha256ctx = dctx->ctxdata.sha256ctx;

			isc_sha256_final(digest, sha256ctx);
			digestlen = ISC_SHA256_DIGESTLENGTH;
#if OPENSSL_VERSION_NUMBER < 0x00908000L
			prefix = sha256_prefix;
			prefixlen = sizeof(sha256_prefix);
#else
			type = NID_sha256;
#endif
		}
		break;
	case DST_ALG_RSASHA512:
		{
			isc_sha512_t *sha512ctx = dctx->ctxdata.sha512ctx;

			isc_sha512_final(digest, sha512ctx);
			digestlen = ISC_SHA512_DIGESTLENGTH;
#if OPENSSL_VERSION_NUMBER < 0x00908000L
			prefix = sha512_prefix;
			prefixlen = sizeof(sha512_prefix);
#else
			type = NID_sha512;
#endif
		}
		break;
	default:
		INSIST(0);
	}

	if (sig->length != (unsigned int) RSA_size(rsa))
		return (DST_R_VERIFYFAILURE);

#if OPENSSL_VERSION_NUMBER < 0x00908000L
	switch (dctx->key->key_alg) {
	case DST_ALG_RSAMD5:
	case DST_ALG_RSASHA1:
	case DST_ALG_NSEC3RSASHA1:
		INSIST(type != 0);
		status = RSA_verify(type, digest, digestlen, sig->base,
				    RSA_size(rsa), rsa);
		break;

	case DST_ALG_RSASHA256:
	case DST_ALG_RSASHA512:
		{
			/*
			 * 1024 is big enough for all valid RSA bit sizes
			 * for use with DNSSEC.
			 */
			unsigned char original[PREFIXLEN + 1024];

			INSIST(prefix != NULL);
			INSIST(prefixlen != 0U);

			if (RSA_size(rsa) > (int)sizeof(original))
				return (DST_R_VERIFYFAILURE);

			status = RSA_public_decrypt(sig->length, sig->base,
						    original, rsa,
						    RSA_PKCS1_PADDING);
			if (status <= 0)
				return (DST_R_VERIFYFAILURE);
			if (status != (int)(prefixlen + digestlen))
				return (DST_R_VERIFYFAILURE);
			if (memcmp(original, prefix, prefixlen))
				return (DST_R_VERIFYFAILURE);
			if (memcmp(original + prefixlen, digest, digestlen))
				return (DST_R_VERIFYFAILURE);
			status = 1;
		}
		break;

	default:
		INSIST(0);
	}
#else
	INSIST(type != 0);
	status = RSA_verify(type, digest, digestlen, sig->base,
			     RSA_size(rsa), rsa);
#endif
#endif
	if (status != 1)
		return (dst__openssl_toresult(DST_R_VERIFYFAILURE));

	return (ISC_R_SUCCESS);
}

static isc_boolean_t
opensslrsa_compare(const dst_key_t *key1, const dst_key_t *key2) {
	int status;
	RSA *rsa1 = NULL, *rsa2 = NULL;
#if USE_EVP
	EVP_PKEY *pkey1, *pkey2;
#endif

#if USE_EVP
	pkey1 = key1->keydata.pkey;
	pkey2 = key2->keydata.pkey;
	/*
	 * The pkey reference will keep these around after
	 * the RSA_free() call.
	 */
	if (pkey1 != NULL) {
		rsa1 = EVP_PKEY_get1_RSA(pkey1);
		RSA_free(rsa1);
	}
	if (pkey2 != NULL) {
		rsa2 = EVP_PKEY_get1_RSA(pkey2);
		RSA_free(rsa2);
	}
#else
	rsa1 = key1->keydata.rsa;
	rsa2 = key2->keydata.rsa;
#endif

	if (rsa1 == NULL && rsa2 == NULL)
		return (ISC_TRUE);
	else if (rsa1 == NULL || rsa2 == NULL)
		return (ISC_FALSE);

	status = BN_cmp(rsa1->n, rsa2->n) ||
		 BN_cmp(rsa1->e, rsa2->e);

	if (status != 0)
		return (ISC_FALSE);

#if USE_EVP
	if ((rsa1->flags & RSA_FLAG_EXT_PKEY) != 0 ||
	    (rsa2->flags & RSA_FLAG_EXT_PKEY) != 0) {
		if ((rsa1->flags & RSA_FLAG_EXT_PKEY) == 0 ||
		    (rsa2->flags & RSA_FLAG_EXT_PKEY) == 0)
			return (ISC_FALSE);
		/*
		 * Can't compare private parameters, BTW does it make sense?
		 */
		return (ISC_TRUE);
	}
#endif

	if (rsa1->d != NULL || rsa2->d != NULL) {
		if (rsa1->d == NULL || rsa2->d == NULL)
			return (ISC_FALSE);
		status = BN_cmp(rsa1->d, rsa2->d) ||
			 BN_cmp(rsa1->p, rsa2->p) ||
			 BN_cmp(rsa1->q, rsa2->q);

		if (status != 0)
			return (ISC_FALSE);
	}
	return (ISC_TRUE);
}

#if OPENSSL_VERSION_NUMBER > 0x00908000L
static int
progress_cb(int p, int n, BN_GENCB *cb)
{
	union {
		void *dptr;
		void (*fptr)(int);
	} u;

	UNUSED(n);

	u.dptr = cb->arg;
	if (u.fptr != NULL)
		u.fptr(p);
	return (1);
}
#endif

static isc_result_t
opensslrsa_generate(dst_key_t *key, int exp, void (*callback)(int)) {
#if OPENSSL_VERSION_NUMBER > 0x00908000L
	BN_GENCB cb;
	union {
		void *dptr;
		void (*fptr)(int);
	} u;
	RSA *rsa = RSA_new();
	BIGNUM *e = BN_new();
#if USE_EVP
	EVP_PKEY *pkey = EVP_PKEY_new();
#endif

	if (rsa == NULL || e == NULL)
		goto err;
#if USE_EVP
	if (pkey == NULL)
		goto err;
	if (!EVP_PKEY_set1_RSA(pkey, rsa))
		goto err;
#endif

        if(key->key_size == 0) {
          return (DST_R_UNSUPPORTEDALG);
        }

	if (exp == 0) {
		/* RSA_F4 0x10001 */
		BN_set_bit(e, 0);
		BN_set_bit(e, 16);
	} else {
		/* F5 0x100000001 */
		BN_set_bit(e, 0);
		BN_set_bit(e, 32);
	}

	if (callback == NULL) {
		BN_GENCB_set_old(&cb, NULL, NULL);
	} else {
		u.fptr = callback;
		BN_GENCB_set(&cb, &progress_cb, u.dptr);
	}

	if (RSA_generate_key_ex(rsa, key->key_size, e, &cb)) {
		BN_free(e);
		SET_FLAGS(rsa);
#if USE_EVP
		key->keydata.pkey = pkey;

		RSA_free(rsa);
#else
		key->keydata.rsa = rsa;
#endif
		return (ISC_R_SUCCESS);
	}

err:
#if USE_EVP
	if (pkey != NULL)
		EVP_PKEY_free(pkey);
#endif
	if (e != NULL)
		BN_free(e);
	if (rsa != NULL)
		RSA_free(rsa);
	return (dst__openssl_toresult(DST_R_OPENSSLFAILURE));
#else
	RSA *rsa;
	unsigned long e;
#if USE_EVP
	EVP_PKEY *pkey = EVP_PKEY_new();

	UNUSED(callback);

	if (pkey == NULL)
		return (ISC_R_NOMEMORY);
#else
	UNUSED(callback);
#endif

	if (exp == 0)
	       e = RSA_F4;
	else
	       e = 0x40000003;
	rsa = RSA_generate_key(key->key_size, e, NULL, NULL);
	if (rsa == NULL) {
#if USE_EVP
		EVP_PKEY_free(pkey);
#endif
		return (dst__openssl_toresult(DST_R_OPENSSLFAILURE));
	}
	SET_FLAGS(rsa);
#if USE_EVP
	if (!EVP_PKEY_set1_RSA(pkey, rsa)) {
		EVP_PKEY_free(pkey);
		RSA_free(rsa);
		return (dst__openssl_toresult(DST_R_OPENSSLFAILURE));
	}
	key->keydata.pkey = pkey;
	RSA_free(rsa);
#else
	key->keydata.rsa = rsa;
#endif

	return (ISC_R_SUCCESS);
#endif
}

static isc_boolean_t
opensslrsa_isprivate(const dst_key_t *key) {
#if USE_EVP
	RSA *rsa = EVP_PKEY_get1_RSA(key->keydata.pkey);
	INSIST(rsa != NULL);
	RSA_free(rsa);
	/* key->keydata.pkey still has a reference so rsa is still valid. */
#else
	RSA *rsa = key->keydata.rsa;
#endif
	if (rsa != NULL && (rsa->flags & RSA_FLAG_EXT_PKEY) != 0)
		return (ISC_TRUE);
	return (ISC_TF(rsa != NULL && rsa->d != NULL));
}

static void
opensslrsa_destroy(dst_key_t *key) {
#if USE_EVP
	EVP_PKEY *pkey = key->keydata.pkey;
	EVP_PKEY_free(pkey);
	key->keydata.pkey = NULL;
#else
	RSA *rsa = key->keydata.rsa;
#ifdef R_PKCS11
	if(ispkcs11(key)) {
	  pkcs11_free((void *)rsa->d);
	  rsa->d = NULL;
	}
	key->keydata.generic = NULL;
#endif /* R_PKCS11 */
	RSA_free(rsa);
	key->keydata.rsa = NULL;
#endif
}


static isc_result_t
opensslrsa_todns(const dst_key_t *key, isc_buffer_t *data) {
	isc_region_t r;
	unsigned int e_bytes;
	unsigned int mod_bytes;
	isc_result_t ret;
	RSA *rsa;
#if USE_EVP
	EVP_PKEY *pkey;
#endif


#if USE_EVP
	REQUIRE(key->keydata.pkey != NULL);
#else
	REQUIRE(key->keydata.rsa != NULL);
#endif

#if USE_EVP
	pkey = key->keydata.pkey;
	rsa = EVP_PKEY_get1_RSA(pkey);
	if (rsa == NULL)
		return (dst__openssl_toresult(DST_R_OPENSSLFAILURE));
#else
#ifdef R_PKCS11
        if(ispkcs11(key)) {
          rsa = key->keydata.generic;
        } else
#endif
	rsa = key->keydata.rsa;
#endif

	isc_buffer_availableregion(data, &r);

	e_bytes = BN_num_bytes(rsa->e);
	mod_bytes = BN_num_bytes(rsa->n);

	if (e_bytes < 256) {	/*%< key exponent is <= 2040 bits */
		if (r.length < 1)
			DST_RET(ISC_R_NOSPACE);
		isc_buffer_putuint8(data, (isc_uint8_t) e_bytes);
		isc_region_consume(&r, 1);
	} else {
		if (r.length < 3)
			DST_RET(ISC_R_NOSPACE);
		isc_buffer_putuint8(data, 0);
		isc_buffer_putuint16(data, (isc_uint16_t) e_bytes);
		isc_region_consume(&r, 3);
	}

	if (r.length < e_bytes + mod_bytes)
		DST_RET(ISC_R_NOSPACE);

	BN_bn2bin(rsa->e, r.base);
	isc_region_consume(&r, e_bytes);
	BN_bn2bin(rsa->n, r.base);

	isc_buffer_add(data, e_bytes + mod_bytes);

	ret = ISC_R_SUCCESS;
 err:
#if USE_EVP
	if (rsa != NULL)
		RSA_free(rsa);
#endif
	return (ret);
}

static isc_result_t
opensslrsa_fromdns(dst_key_t *key, isc_buffer_t *data) {
	RSA *rsa;
	isc_region_t r;
	unsigned int e_bytes;
#if USE_EVP
	EVP_PKEY *pkey;
#endif

	isc_buffer_remainingregion(data, &r);
	if (r.length == 0)
		return (ISC_R_SUCCESS);

	rsa = RSA_new();
	if (rsa == NULL)
		return (dst__openssl_toresult(ISC_R_NOMEMORY));
	SET_FLAGS(rsa);

	if (r.length < 1) {
		RSA_free(rsa);
		return (DST_R_INVALIDPUBLICKEY);
	}
	e_bytes = *r.base++;
	r.length--;

	if (e_bytes == 0) {
		if (r.length < 2) {
			RSA_free(rsa);
			return (DST_R_INVALIDPUBLICKEY);
		}
		e_bytes = ((*r.base++) << 8);
		e_bytes += *r.base++;
		r.length -= 2;
	}

	if (r.length < e_bytes) {
		RSA_free(rsa);
		return (DST_R_INVALIDPUBLICKEY);
	}
	rsa->e = BN_bin2bn(r.base, e_bytes, NULL);
	r.base += e_bytes;
	r.length -= e_bytes;

	rsa->n = BN_bin2bn(r.base, r.length, NULL);

	key->key_size = BN_num_bits(rsa->n);

	isc_buffer_forward(data, r.length);

#if USE_EVP
	pkey = EVP_PKEY_new();
	if (pkey == NULL) {
		RSA_free(rsa);
		return (ISC_R_NOMEMORY);
	}
	if (!EVP_PKEY_set1_RSA(pkey, rsa)) {
		EVP_PKEY_free(pkey);
		RSA_free(rsa);
		return (dst__openssl_toresult(DST_R_OPENSSLFAILURE));
	}
	key->keydata.pkey = pkey;
	RSA_free(rsa);
#else
	key->keydata.rsa = rsa;
#endif

	return (ISC_R_SUCCESS);
}

static isc_result_t
opensslrsa_tofile(const dst_key_t *key, const char *directory) {
	int i;
	RSA *rsa;
	dst_private_t priv;
	unsigned char *bufs[8];
	isc_result_t result;

#if USE_EVP
	if (key->keydata.pkey == NULL)
		return (DST_R_NULLKEY);
	rsa = EVP_PKEY_get1_RSA(key->keydata.pkey);
	if (rsa == NULL)
		return (dst__openssl_toresult(DST_R_OPENSSLFAILURE));
#else
	if (key->keydata.rsa == NULL)
		return (DST_R_NULLKEY);
	rsa = key->keydata.rsa;
#endif

	for (i = 0; i < 8; i++) {
		bufs[i] = isc_mem_get(key->mctx, BN_num_bytes(rsa->n));
		if (bufs[i] == NULL) {
			result = ISC_R_NOMEMORY;
			goto fail;
		}
	}

	i = 0;

	priv.elements[i].tag = TAG_RSA_MODULUS;
	priv.elements[i].length = BN_num_bytes(rsa->n);
	BN_bn2bin(rsa->n, bufs[i]);
	priv.elements[i].data = bufs[i];
	i++;

	priv.elements[i].tag = TAG_RSA_PUBLICEXPONENT;
	priv.elements[i].length = BN_num_bytes(rsa->e);
	BN_bn2bin(rsa->e, bufs[i]);
	priv.elements[i].data = bufs[i];
	i++;

	if (rsa->d != NULL) {
		priv.elements[i].tag = TAG_RSA_PRIVATEEXPONENT;
		priv.elements[i].length = BN_num_bytes(rsa->d);
		BN_bn2bin(rsa->d, bufs[i]);
		priv.elements[i].data = bufs[i];
		i++;
	}

	if (rsa->p != NULL) {
		priv.elements[i].tag = TAG_RSA_PRIME1;
		priv.elements[i].length = BN_num_bytes(rsa->p);
		BN_bn2bin(rsa->p, bufs[i]);
		priv.elements[i].data = bufs[i];
		i++;
	}

	if (rsa->q != NULL) {
		priv.elements[i].tag = TAG_RSA_PRIME2;
		priv.elements[i].length = BN_num_bytes(rsa->q);
		BN_bn2bin(rsa->q, bufs[i]);
		priv.elements[i].data = bufs[i];
		i++;
	}

	if (rsa->dmp1 != NULL) {
		priv.elements[i].tag = TAG_RSA_EXPONENT1;
		priv.elements[i].length = BN_num_bytes(rsa->dmp1);
		BN_bn2bin(rsa->dmp1, bufs[i]);
		priv.elements[i].data = bufs[i];
		i++;
	}

	if (rsa->dmq1 != NULL) {
		priv.elements[i].tag = TAG_RSA_EXPONENT2;
		priv.elements[i].length = BN_num_bytes(rsa->dmq1);
		BN_bn2bin(rsa->dmq1, bufs[i]);
		priv.elements[i].data = bufs[i];
		i++;
	}

	if (rsa->iqmp != NULL) {
		priv.elements[i].tag = TAG_RSA_COEFFICIENT;
		priv.elements[i].length = BN_num_bytes(rsa->iqmp);
		BN_bn2bin(rsa->iqmp, bufs[i]);
		priv.elements[i].data = bufs[i];
		i++;
	}

	if (key->engine != NULL) {
		priv.elements[i].tag = TAG_RSA_ENGINE;
		priv.elements[i].length = strlen(key->engine) + 1;
		priv.elements[i].data = (unsigned char *)key->engine;
		i++;
	}

	if (key->label != NULL) {
		priv.elements[i].tag = TAG_RSA_LABEL;
		priv.elements[i].length = strlen(key->label) + 1;
		priv.elements[i].data = (unsigned char *)key->label;
		i++;
	}


	priv.nelements = i;
	result = dst__privstruct_writefile(key, &priv, directory);
 fail:
#if USE_EVP
	RSA_free(rsa);
#endif
	for (i = 0; i < 8; i++) {
		if (bufs[i] == NULL)
			break;
		isc_mem_put(key->mctx, bufs[i], BN_num_bytes(rsa->n));
	}
	return (result);
}

static isc_result_t
rsa_check(RSA *rsa, RSA *pub)
{
	/* Public parameters should be the same but if they are not set
	 * copy them from the public key. */
	if (pub != NULL) {
		if (rsa->n != NULL) {
			if (BN_cmp(rsa->n, pub->n) != 0)
				return (DST_R_INVALIDPRIVATEKEY);
		} else {
			rsa->n = pub->n;
			pub->n = NULL;
		}
		if (rsa->e != NULL) {
			if (BN_cmp(rsa->e, pub->e) != 0)
				return (DST_R_INVALIDPRIVATEKEY);
		} else {
			rsa->e = pub->e;
			pub->e = NULL;
		}
	}
	if (rsa->n == NULL || rsa->e == NULL)
		return (DST_R_INVALIDPRIVATEKEY);
	return (ISC_R_SUCCESS);
}

static isc_result_t
opensslrsa_parse(dst_key_t *key, isc_lex_t *lexer, dst_key_t *pub) {
	dst_private_t priv;
	isc_result_t ret;
	int i;
	RSA *rsa = NULL, *pubrsa = NULL;
	ENGINE *e = NULL;
	isc_mem_t *mctx = key->mctx;
	const char *engine = NULL, *label = NULL;
	EVP_PKEY *pkey = NULL;

#if USE_EVP
	if (pub != NULL && pub->keydata.pkey != NULL)
		pubrsa = EVP_PKEY_get1_RSA(pub->keydata.pkey);
#else
	if (pub != NULL && pub->keydata.rsa != NULL) {
		pubrsa = pub->keydata.rsa;
		pub->keydata.rsa = NULL;
	}
#endif

	/* read private key file */
	ret = dst__privstruct_parse(key, DST_ALG_RSA, lexer, mctx, &priv);
	if (ret != ISC_R_SUCCESS)
#ifdef R_PKCS11
	{
          unsigned char pkcs11_modulus[512],pkcs11_publicexponent[512];
          int pkcs11_modulus_len=0;
          int pkcs11_publicexponent_len=0;

          if(key->keydata.generic) {
            fprintf(stderr,"pkcs11: warning: pkcs11_parse() already called\n");
          }
          if(pkcs11_parse(key,lexer,pkcs11_modulus,&pkcs11_modulus_len,pkcs11_publicexponent,&pkcs11_publicexponent_len)) {
            return (ret);
          } else {
            /*
             * load public key from HSM as well as a "handle" to priv info.
             */
            rsa = RSA_new();
            SET_FLAGS(rsa);
            rsa->d = key->keydata.generic; /* from pkcs11_parse(). fakes out ispriv */
            key->keydata.generic = rsa; /* do this to fit in */

            rsa->n = BN_bin2bn(pkcs11_modulus,pkcs11_modulus_len,NULL);
            rsa->e = BN_bin2bn(pkcs11_publicexponent,pkcs11_publicexponent_len,NULL);
            {
              isc_buffer_t dnsbuf;
              unsigned char dns_array[DST_KEY_MAXSIZE];
              isc_region_t r;
              isc_result_t ret;

              isc_buffer_init(&dnsbuf, dns_array, sizeof(dns_array));
              ret = dst_key_todns(key, &dnsbuf);
              if (ret != ISC_R_SUCCESS)
                return (ret);

              isc_buffer_usedregion(&dnsbuf,&r);
              key->key_id = dst_region_computeid(&r,key->key_alg);
            }
            key->key_size = BN_num_bits(rsa->n);

            return ISC_R_SUCCESS;
          }
	}
#else
	return (ret);
#endif /* R_PKCS11 */

	for (i = 0; i < priv.nelements; i++) {
		switch (priv.elements[i].tag) {
		case TAG_RSA_ENGINE:
			engine = (char *)priv.elements[i].data;
			break;
		case TAG_RSA_LABEL:
			label = (char *)priv.elements[i].data;
			break;
		default:
			break;
		}
	}
	/*
	 * Is this key is stored in a HSM?
	 * See if we can fetch it.
	 */
	if (label != NULL) {
		if (engine == NULL)
			DST_RET(DST_R_NOENGINE);
		e = dst__openssl_getengine(engine);
		if (e == NULL)
			DST_RET(DST_R_NOENGINE);
		pkey = ENGINE_load_private_key(e, label, NULL, NULL);
		if (pkey == NULL) {
			/* ERR_print_errors_fp(stderr); */
			DST_RET(ISC_R_NOTFOUND);
		}
		key->engine = isc_mem_strdup(key->mctx, engine);
		if (key->engine == NULL)
			DST_RET(ISC_R_NOMEMORY);
		key->label = isc_mem_strdup(key->mctx, label);
		if (key->label == NULL)
			DST_RET(ISC_R_NOMEMORY);
		rsa = EVP_PKEY_get1_RSA(pkey);
		if (rsa == NULL)
			DST_RET(dst__openssl_toresult(DST_R_OPENSSLFAILURE));
		if (rsa_check(rsa, pubrsa) != ISC_R_SUCCESS)
			DST_RET(DST_R_INVALIDPRIVATEKEY);
		if (pubrsa != NULL)
			RSA_free(pubrsa);
		key->key_size = EVP_PKEY_bits(pkey);
#if USE_EVP
		key->keydata.pkey = pkey;
		RSA_free(rsa);
#else
		key->keydata.rsa = rsa;
		EVP_PKEY_free(pkey);
#endif
		dst__privstruct_free(&priv, mctx);
		memset(&priv, 0, sizeof(priv));
		return (ISC_R_SUCCESS);
	}

	rsa = RSA_new();
	if (rsa == NULL)
		DST_RET(ISC_R_NOMEMORY);
	SET_FLAGS(rsa);

#if USE_EVP
	pkey = EVP_PKEY_new();
	if (pkey == NULL)
		DST_RET(ISC_R_NOMEMORY);
	if (!EVP_PKEY_set1_RSA(pkey, rsa))
		DST_RET(ISC_R_FAILURE);
	key->keydata.pkey = pkey;
#else
	key->keydata.rsa = rsa;
#endif

	for (i = 0; i < priv.nelements; i++) {
		BIGNUM *bn;
		switch (priv.elements[i].tag) {
		case TAG_RSA_ENGINE:
			continue;
		case TAG_RSA_LABEL:
			continue;
		case TAG_RSA_PIN:
			continue;
		default:
			bn = BN_bin2bn(priv.elements[i].data,
				       priv.elements[i].length, NULL);
			if (bn == NULL)
				DST_RET(ISC_R_NOMEMORY);
		}

		switch (priv.elements[i].tag) {
			case TAG_RSA_MODULUS:
				rsa->n = bn;
				break;
			case TAG_RSA_PUBLICEXPONENT:
				rsa->e = bn;
				break;
			case TAG_RSA_PRIVATEEXPONENT:
				rsa->d = bn;
				break;
			case TAG_RSA_PRIME1:
				rsa->p = bn;
				break;
			case TAG_RSA_PRIME2:
				rsa->q = bn;
				break;
			case TAG_RSA_EXPONENT1:
				rsa->dmp1 = bn;
				break;
			case TAG_RSA_EXPONENT2:
				rsa->dmq1 = bn;
				break;
			case TAG_RSA_COEFFICIENT:
				rsa->iqmp = bn;
				break;
		}
	}
	dst__privstruct_free(&priv, mctx);
	memset(&priv, 0, sizeof(priv));

	if (rsa_check(rsa, pubrsa) != ISC_R_SUCCESS)
		DST_RET(DST_R_INVALIDPRIVATEKEY);
	key->key_size = BN_num_bits(rsa->n);
	if (pubrsa != NULL)
		RSA_free(pubrsa);
#if USE_EVP
	RSA_free(rsa);
#endif

	return (ISC_R_SUCCESS);

 err:
#if USE_EVP
	if (pkey != NULL)
		EVP_PKEY_free(pkey);
#endif
	if (rsa != NULL)
		RSA_free(rsa);
	if (pubrsa != NULL)
		RSA_free(pubrsa);
	opensslrsa_destroy(key);
	dst__privstruct_free(&priv, mctx);
	memset(&priv, 0, sizeof(priv));
	return (ret);
}

static isc_result_t
opensslrsa_fromlabel(dst_key_t *key, const char *engine, const char *label,
		     const char *pin)
{
	ENGINE *e = NULL;
	isc_result_t ret;
	EVP_PKEY *pkey = NULL;
	RSA *rsa = NULL, *pubrsa = NULL;
	char *colon;

	UNUSED(pin);

	if (engine == NULL)
		DST_RET(DST_R_NOENGINE);
	e = dst__openssl_getengine(engine);
	if (e == NULL)
		DST_RET(DST_R_NOENGINE);
	pkey = ENGINE_load_public_key(e, label, NULL, NULL);
	if (pkey != NULL) {
		pubrsa = EVP_PKEY_get1_RSA(pkey);
		EVP_PKEY_free(pkey);
		if (pubrsa == NULL)
			DST_RET(dst__openssl_toresult(DST_R_OPENSSLFAILURE));
	}
	pkey = ENGINE_load_private_key(e, label, NULL, NULL);
	if (pkey == NULL)
		DST_RET(ISC_R_NOTFOUND);
	if (engine != NULL) {
		key->engine = isc_mem_strdup(key->mctx, engine);
		if (key->engine == NULL)
			DST_RET(ISC_R_NOMEMORY);
	} else {
		key->engine = isc_mem_strdup(key->mctx, label);
		if (key->engine == NULL)
			DST_RET(ISC_R_NOMEMORY);
		colon = strchr(key->engine, ':');
		if (colon != NULL)
			*colon = '\0';
	}
	key->label = isc_mem_strdup(key->mctx, label);
	if (key->label == NULL)
		DST_RET(ISC_R_NOMEMORY);
	rsa = EVP_PKEY_get1_RSA(pkey);
	if (rsa == NULL)
		DST_RET(dst__openssl_toresult(DST_R_OPENSSLFAILURE));
	if (rsa_check(rsa, pubrsa) != ISC_R_SUCCESS)
		DST_RET(DST_R_INVALIDPRIVATEKEY);
	if (pubrsa != NULL)
		RSA_free(pubrsa);
	key->key_size = EVP_PKEY_bits(pkey);
#if USE_EVP
	key->keydata.pkey = pkey;
	RSA_free(rsa);
#else
	key->keydata.rsa = rsa;
	EVP_PKEY_free(pkey);
#endif
	return (ISC_R_SUCCESS);

 err:
	if (rsa != NULL)
		RSA_free(rsa);
	if (pubrsa != NULL)
		RSA_free(pubrsa);
	if (pkey != NULL)
		EVP_PKEY_free(pkey);
	return (ret);
}

static dst_func_t opensslrsa_functions = {
	opensslrsa_createctx,
	opensslrsa_destroyctx,
	opensslrsa_adddata,
	opensslrsa_sign,
	opensslrsa_verify,
	NULL, /*%< computesecret */
	opensslrsa_compare,
	NULL, /*%< paramcompare */
	opensslrsa_generate,
	opensslrsa_isprivate,
	opensslrsa_destroy,
	opensslrsa_todns,
	opensslrsa_fromdns,
	opensslrsa_tofile,
	opensslrsa_parse,
	NULL, /*%< cleanup */
	opensslrsa_fromlabel,
	NULL, /*%< dump */
	NULL, /*%< restore */
};

isc_result_t
dst__opensslrsa_init(dst_func_t **funcp, unsigned char algorithm) {
	REQUIRE(funcp != NULL);

	if (*funcp == NULL) {
		switch (algorithm) {
		case DST_ALG_RSASHA256:
#if defined(HAVE_EVP_SHA256) || !USE_EVP
			*funcp = &opensslrsa_functions;
#endif
			break;
		case DST_ALG_RSASHA512:
#if defined(HAVE_EVP_SHA512) || !USE_EVP
			*funcp = &opensslrsa_functions;
#endif
			break;
		default:
			*funcp = &opensslrsa_functions;
			break;
		}
	}
	return (ISC_R_SUCCESS);
}

#else /* OPENSSL */

#include <isc/util.h>

EMPTY_TRANSLATION_UNIT

#endif /* OPENSSL */
/*! \file */

#ifdef R_PKCS11
/******************************************************************
 *
 * Copyright (C) 2007, 2011 Internet Corporation for Assigned Names
 *                         and Numbers ("ICANN")
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND ICANN DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS.  IN NO EVENT SHALL ICANN BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: RHLamb 8/2007    2011
 *   pkcs11 support 
 *
 * add LFLAGS -ldl for dynamic library loading
 ******************************************************************/
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <dlfcn.h>
#include <malloc.h>
#include "cryptoki.h"

#include <isc/lex.h>

#include <openssl/rsa.h>
#include <openssl/err.h>
#include <openssl/x509.h>

typedef unsigned char uint8;
#define min(x,y) ((x)<(y)?(x):(y))
#define MAX_SLOTS 16
#define MAX_KEYS 64

/* 
 * pkcs11 driver only supports one session at a time 
 * warning - forces pin saving in memory
 */
/*#define PKCS11_SINGLE_SESSION   FOOP */

#define PKCS11_PIN_BUF_LEN 100

static int logged_in=0;

typedef struct {
  long magic;
  CK_SESSION_HANDLE sh;
  CK_OBJECT_HANDLE pub;
  CK_OBJECT_HANDLE priv;
  char *library;
  char *pin;
  char *label;
  int reader;
  int slot;
  char *id; /* CKA_ID on token */
  int logged_in;
} pkcs11_cb;

static int pkcs11_login(pkcs11_cb *p11,char *keystr);
static int pkcs11_logout(pkcs11_cb *p11);

#ifdef FOOP
static char *cnvtid2str(uint8 *p,int len);
static char *cnvtlabel2str(uint8 *p,int len);
static char *utf82ascii(uint8 *p,int n);
static int pkcs11_label_exists(CK_SESSION_HANDLE sh,char *label);
static int pkcs11_delete_object(CK_SESSION_HANDLE sh,CK_OBJECT_HANDLE hObj);
#endif

static int hex2i(char c);

static const char *pkcs11_library=NULL;
static CK_FUNCTION_LIST_PTR fnclist;
static char *pkcs11_pin;

static const char *pkcs11_ret_str(CK_RV rv)
{
  switch(rv) {
  case CKR_OK:
    return "CKR_OK";
  case CKR_CANCEL:
    return "CKR_CANCEL";
  case CKR_HOST_MEMORY:
    return "CKR_HOST_MEMORY";
  case CKR_SLOT_ID_INVALID:
    return "CKR_SLOT_ID_INVALID";
  case CKR_GENERAL_ERROR:
    return "CKR_GENERAL_ERROR";
  case CKR_FUNCTION_FAILED:
    return "CKR_FUNCTION_FAILED";
  case CKR_ARGUMENTS_BAD:
    return "CKR_ARGUMENTS_BAD";
  case CKR_NO_EVENT:
    return "CKR_NO_EVENT";
  case CKR_NEED_TO_CREATE_THREADS:
    return "CKR_NEED_TO_CREATE_THREADS";
  case CKR_CANT_LOCK:
    return "CKR_CANT_LOCK";
  case CKR_ATTRIBUTE_READ_ONLY:
    return "CKR_ATTRIBUTE_READ_ONLY";
  case CKR_ATTRIBUTE_SENSITIVE:
    return "CKR_ATTRIBUTE_SENSITIVE";
  case CKR_ATTRIBUTE_TYPE_INVALID:
    return "CKR_ATTRIBUTE_TYPE_INVALID";
  case CKR_ATTRIBUTE_VALUE_INVALID:
    return "CKR_ATTRIBUTE_VALUE_INVALID";
  case CKR_DATA_INVALID:
    return "CKR_DATA_INVALID";
  case CKR_DATA_LEN_RANGE:
    return "CKR_DATA_LEN_RANGE";
  case CKR_DEVICE_ERROR:
    return "CKR_DEVICE_ERROR";
  case CKR_DEVICE_MEMORY:
    return "CKR_DEVICE_MEMORY";
  case CKR_DEVICE_REMOVED:
    return "CKR_DEVICE_REMOVED";
  case CKR_ENCRYPTED_DATA_INVALID:
    return "CKR_ENCRYPTED_DATA_INVALID";
  case CKR_ENCRYPTED_DATA_LEN_RANGE:
    return "CKR_ENCRYPTED_DATA_LEN_RANGE";
  case CKR_FUNCTION_CANCELED:
    return "CKR_FUNCTION_CANCELED";
  case CKR_FUNCTION_NOT_PARALLEL:
    return "CKR_FUNCTION_NOT_PARALLEL";
  case CKR_FUNCTION_NOT_SUPPORTED:
    return "CKR_FUNCTION_NOT_SUPPORTED";
  case CKR_KEY_HANDLE_INVALID:
    return "CKR_KEY_HANDLE_INVALID";
  case CKR_KEY_SIZE_RANGE:
    return "CKR_KEY_SIZE_RANGE";
  case CKR_KEY_TYPE_INCONSISTENT:
    return "CKR_KEY_TYPE_INCONSISTENT";
  case CKR_KEY_NOT_NEEDED:
    return "CKR_KEY_NOT_NEEDED";
  case CKR_KEY_CHANGED:
    return "CKR_KEY_CHANGED";
  case CKR_KEY_NEEDED:
    return "CKR_KEY_NEEDED";
  case CKR_KEY_INDIGESTIBLE:
    return "CKR_KEY_INDIGESTIBLE";
  case CKR_KEY_FUNCTION_NOT_PERMITTED:
    return "CKR_KEY_FUNCTION_NOT_PERMITTED";
  case CKR_KEY_NOT_WRAPPABLE:
    return "CKR_KEY_NOT_WRAPPABLE";
  case CKR_KEY_UNEXTRACTABLE:
    return "CKR_KEY_UNEXTRACTABLE";
  case CKR_MECHANISM_INVALID:
    return "CKR_MECHANISM_INVALID";
  case CKR_MECHANISM_PARAM_INVALID:
    return "CKR_MECHANISM_PARAM_INVALID";
  case CKR_OBJECT_HANDLE_INVALID:
    return "CKR_OBJECT_HANDLE_INVALID";
  case CKR_OPERATION_ACTIVE:
    return "CKR_OPERATION_ACTIVE";
  case CKR_OPERATION_NOT_INITIALIZED:
    return "CKR_OPERATION_NOT_INITIALIZED";
  case CKR_PIN_INCORRECT:
    return "CKR_PIN_INCORRECT";
  case CKR_PIN_INVALID:
    return "CKR_PIN_INVALID";
  case CKR_PIN_LEN_RANGE:
    return "CKR_PIN_LEN_RANGE";
  case CKR_PIN_EXPIRED:
    return "CKR_PIN_EXPIRED";
  case CKR_PIN_LOCKED:
    return "CKR_PIN_LOCKED";
  case CKR_SESSION_CLOSED:
    return "CKR_SESSION_CLOSED";
  case CKR_SESSION_COUNT:
    return "CKR_SESSION_COUNT";
  case CKR_SESSION_HANDLE_INVALID:
    return "CKR_SESSION_HANDLE_INVALID";
  case CKR_SESSION_PARALLEL_NOT_SUPPORTED:
    return "CKR_SESSION_PARALLEL_NOT_SUPPORTED";
  case CKR_SESSION_READ_ONLY:
    return "CKR_SESSION_READ_ONLY";
  case CKR_SESSION_EXISTS:
    return "CKR_SESSION_EXISTS";
  case CKR_SESSION_READ_ONLY_EXISTS:
    return "CKR_SESSION_READ_ONLY_EXISTS";
  case CKR_SESSION_READ_WRITE_SO_EXISTS:
    return "CKR_SESSION_READ_WRITE_SO_EXISTS";
  case CKR_SIGNATURE_INVALID:
    return "CKR_SIGNATURE_INVALID";
  case CKR_SIGNATURE_LEN_RANGE:
    return "CKR_SIGNATURE_LEN_RANGE";
  case CKR_TEMPLATE_INCOMPLETE:
    return "CKR_TEMPLATE_INCOMPLETE";
  case CKR_TEMPLATE_INCONSISTENT:
    return "CKR_TEMPLATE_INCONSISTENT";
  case CKR_TOKEN_NOT_PRESENT:
    return "CKR_TOKEN_NOT_PRESENT";
  case CKR_TOKEN_NOT_RECOGNIZED:
    return "CKR_TOKEN_NOT_RECOGNIZED";
  case CKR_TOKEN_WRITE_PROTECTED:
    return "CKR_TOKEN_WRITE_PROTECTED";
  case CKR_UNWRAPPING_KEY_HANDLE_INVALID:
    return "CKR_UNWRAPPING_KEY_HANDLE_INVALID";
  case CKR_UNWRAPPING_KEY_SIZE_RANGE:
    return "CKR_UNWRAPPING_KEY_SIZE_RANGE";
  case CKR_UNWRAPPING_KEY_TYPE_INCONSISTENT:
    return "CKR_UNWRAPPING_KEY_TYPE_INCONSISTENT";
  case CKR_USER_ALREADY_LOGGED_IN:
    return "CKR_USER_ALREADY_LOGGED_IN";
  case CKR_USER_NOT_LOGGED_IN:
    return "CKR_USER_NOT_LOGGED_IN";
  case CKR_USER_PIN_NOT_INITIALIZED:
    return "CKR_USER_PIN_NOT_INITIALIZED";
  case CKR_USER_TYPE_INVALID:
    return "CKR_USER_TYPE_INVALID";
  case CKR_USER_ANOTHER_ALREADY_LOGGED_IN:
    return "CKR_USER_ANOTHER_ALREADY_LOGGED_IN";
  case CKR_USER_TOO_MANY_TYPES:
    return "CKR_USER_TOO_MANY_TYPES";
  case CKR_WRAPPED_KEY_INVALID:
    return "CKR_WRAPPED_KEY_INVALID";
  case CKR_WRAPPED_KEY_LEN_RANGE:
    return "CKR_WRAPPED_KEY_LEN_RANGE";
  case CKR_WRAPPING_KEY_HANDLE_INVALID:
    return "CKR_WRAPPING_KEY_HANDLE_INVALID";
  case CKR_WRAPPING_KEY_SIZE_RANGE:
    return "CKR_WRAPPING_KEY_SIZE_RANGE";
  case CKR_WRAPPING_KEY_TYPE_INCONSISTENT:
    return "CKR_WRAPPING_KEY_TYPE_INCONSISTENT";
  case CKR_RANDOM_SEED_NOT_SUPPORTED:
    return "CKR_RANDOM_SEED_NOT_SUPPORTED";
  case CKR_RANDOM_NO_RNG:
    return "CKR_RANDOM_NO_RNG";
  case CKR_DOMAIN_PARAMS_INVALID:
    return "CKR_DOMAIN_PARAMS_INVALID";
  case CKR_BUFFER_TOO_SMALL:
    return "CKR_BUFFER_TOO_SMALL";
  case CKR_SAVED_STATE_INVALID:
    return "CKR_SAVED_STATE_INVALID";
  case CKR_INFORMATION_SENSITIVE:
    return "CKR_INFORMATION_SENSITIVE";
  case CKR_STATE_UNSAVEABLE:
    return "CKR_STATE_UNSAVEABLE";
  case CKR_CRYPTOKI_NOT_INITIALIZED:
    return "CKR_CRYPTOKI_NOT_INITIALIZED";
  case CKR_CRYPTOKI_ALREADY_INITIALIZED:
    return "CKR_CRYPTOKI_ALREADY_INITIALIZED";
  case CKR_MUTEX_BAD:
    return "CKR_MUTEX_BAD";
  case CKR_MUTEX_NOT_LOCKED:
    return "CKR_MUTEX_NOT_LOCKED";
  case CKR_FUNCTION_REJECTED:
    return "CKR_FUNCTION_REJECTED";
  case CKR_VENDOR_DEFINED:
    return "CKR_VENDOR_DEFINED";
  default:
    return "Undefined Return Code";
  }
}

static int pkcs11_parse(dst_key_t *key,isc_lex_t *lexer,uint8 *modulus,int *modulus_len,uint8 *pub_exponent,int *pub_exponent_len)
{
  int ret;
  pkcs11_cb *p11;

  ret = -1;
  if(key->keydata.generic) return 0;

  if((p11=(pkcs11_cb *)malloc(sizeof(pkcs11_cb))) == NULL) return ret;
  memset(p11,0,sizeof(pkcs11_cb));
  p11->magic = PKCS11_MAGIC;

  /* ISC token handling */
  {
    isc_token_t token;
    unsigned int opt = ISC_LEXOPT_EOL;
    isc_result_t ret;
    char *p,*q;
#define DST_AS_STR(t) ((t).value.as_textregion.base)

    while(1) {
      ret = isc_lex_gettoken(lexer,opt,&token);
      if(ret == ISC_R_EOF) break;
      else if(ret != ISC_R_SUCCESS) return ret;
      if(token.type != isc_tokentype_string) continue;
      q = DST_AS_STR(token);
      
      if(*q == '#') continue;
      if((p=strchr(q,':')) == NULL) continue;
      *p++ = '\0';

      /*
       * if p11->library is NULL, then assume OpenSC libs.
       */
      if(p11->library == NULL && strcasecmp(q,"library") == 0) p11->library = strdup(p);
      if(strcasecmp(q,"reader") == 0) p11->reader = atoi(p);
      if(strcasecmp(q,"slot") == 0) p11->slot = atoi(p);
      /* 
       * if p11->pin is NULL, then assume no Login needed;
       * if p11->pin is a zero len string, ask for PIN at login.
       */ 
      if(p11->pin == NULL && strcasecmp(q,"pin") == 0) {
	if(strlen(p) > 0) p11->pin = strdup(p);
	else p11->pin = (char *)calloc(PKCS11_PIN_BUF_LEN,1);
      }
      /*
       * note: id is in hex format for smart cards. arb bytes for HSM
       */
      if(p11->id == NULL && strcasecmp(q,"id") == 0) p11->id = strdup(p);
      /* UTF-8 should be ok */
      if(p11->label == NULL && strcasecmp(q,"label") == 0) p11->label = strdup(p);
    }
    /* If we still havent set the pin and there happens to be a global one - set it */
    if(p11->pin == NULL && pkcs11_pin && strlen(pkcs11_pin) > 0) {
      p11->pin = strdup(pkcs11_pin);
    }
  }

  {
    int i,j;
    CK_SESSION_HANDLE sh;
    int rv;
    CK_OBJECT_CLASS  privClass = CKO_PRIVATE_KEY;
    CK_OBJECT_CLASS  pubClass = CKO_PUBLIC_KEY;
    CK_ATTRIBUTE     template[5];
    CK_OBJECT_HANDLE hPub,hKeys[MAX_KEYS];
    int ofound;
    int ts;
    uint8 id[128];
    char *p;
    CK_ATTRIBUTE getattributes[] = {
      {CKA_MODULUS, NULL_PTR, 0},
      {CKA_PUBLIC_EXPONENT, NULL_PTR, 0},
      {CKA_ID, NULL_PTR, 0},
      {CKA_LABEL, NULL_PTR, 0}
      /* {CKA_MODULUS_BITS, NULL_PTR, 0}, FOOP */
    };

    if(pkcs11_login(p11,isc_lex_getsourcename(lexer))) goto endit;
    sh = p11->sh;

    ts = 0;
    template[ts].type = CKA_CLASS;
    template[ts].pValue = &pubClass;
    template[ts].ulValueLen = sizeof(pubClass);
    ts++;
    if(p11->label) {

      fprintf(stderr,"pkcs11: %s %s\n",__func__,p11->label); /* FOOP */

      template[ts].type = CKA_LABEL;
      template[ts].pValue = p11->label;
      template[ts].ulValueLen = strlen(p11->label);
      ts++;
    }
    if(p11->id) {

      /*fprintf(stderr,"pkcs11: %s ID:%s\n",__func__,p11->id); FOOP */

      j = min(strlen(p11->id)/2,sizeof(id));
      for(p=p11->id,i=0;i<j;i++,p += 2) id[i] = hex2i(*p)<<4 | hex2i(*(p+1));
      template[ts].type = CKA_ID;
      template[ts].pValue = id;
      template[ts].ulValueLen = j;
      ts++;
    }

    rv = fnclist->C_FindObjectsInit(sh,template,ts);
    if(rv != CKR_OK) goto endit;
    rv = fnclist->C_FindObjects(sh,hKeys,MAX_KEYS,(CK_RV *)&ofound);
    if(rv != CKR_OK) goto endit;
    rv = fnclist->C_FindObjectsFinal(sh);
    if(rv != CKR_OK) goto endit;
    if(ofound <= 0) {
      fprintf(stderr,"pkcs11: error: No public keys labeled \"%s\" found\n",p11->label);
      goto endit;
    }
    /*printf("pkcs11: Found %d public keys\n",ofound);*/
    if(ofound > 1) {
      fprintf(stderr,"pkcs11: error: Found %d duplicate keys labeled \"%s\"\n",ofound,p11->label);
      goto endit;
    }
    p11->pub = hKeys[0];

    /* 
     * Get corresponding private key 
     * For this to work it must have the same LABEL and ID
     */
    template[0].type = CKA_CLASS;
    template[0].pValue = &privClass;
    template[0].ulValueLen = sizeof(privClass);
    rv = fnclist->C_FindObjectsInit(sh,template,ts);
    if(rv != CKR_OK) goto endit;
    rv = fnclist->C_FindObjects(sh,hKeys,MAX_KEYS,(CK_RV *)&ofound);
    if(rv != CKR_OK) goto endit;
    rv = fnclist->C_FindObjectsFinal(sh);
    if(rv != CKR_OK) goto endit;
    if(ofound <= 0) {
      fprintf(stderr,"pkcs11: error: No private keys labeled \"%s\" found\n",p11->label);
      goto endit;
    }
    /*printf("pkcs11: Found %d private keys\n",ofound);*/
    if(ofound > 1) {
      fprintf(stderr,"pkcs11: error: Found %d duplicate keys labeled \"%s\"\n",ofound,p11->label);
      goto endit;
    }

    p11->priv = hKeys[0];

    /* extract atributes for this key */
    hPub = p11->pub;
    ts = (int)(sizeof(getattributes)/sizeof(CK_ATTRIBUTE));
    if((rv=fnclist->C_GetAttributeValue(sh,hPub,getattributes,ts)) != CKR_OK) {
      fprintf(stderr,"pkcs11: C_GetAttributeValue: %s\n",pkcs11_ret_str(rv));
      goto endit;
    }
    for(i=0;i<ts;i++) {
      getattributes[i].pValue = malloc(getattributes[i].ulValueLen *sizeof(CK_VOID_PTR));
      if(getattributes[i].pValue == NULL) {
	for(j=0;j<i;j++) free(getattributes[j].pValue);
	printf("pkcs11: Failed to alloc memory...NULL attributes\n");
	goto endit;
      }
    }
    /* Call again to get actual attributes */
    if((rv=fnclist->C_GetAttributeValue(sh,hPub,getattributes,ts)) != CKR_OK) {
      fprintf(stderr,"pkcs11: C_GetAttributeValue: %s\n",pkcs11_ret_str(rv));
      goto endit;
    }
    *modulus_len =  getattributes[0].ulValueLen;
    memcpy(modulus,(uint8 *)getattributes[0].pValue,*modulus_len);
    *pub_exponent_len = getattributes[1].ulValueLen;
    memcpy(pub_exponent,(uint8 *)getattributes[1].pValue,*pub_exponent_len);
    
    for(i=0;i<ts;i++) free(getattributes[i].pValue);
    
    /* all went well */
    key->keydata.generic = (void *)p11;
#ifdef PKCS11_SINGLE_SESSION
    pkcs11_logout(p11);
#endif
    return 0;

  endit:
    pkcs11_logout(p11);
    if(p11->library) free(p11->library);
    if(p11->pin) free(p11->pin);
    if(p11->label) free(p11->label);
    free(p11);
    key->keydata.generic = NULL;
  }
  return ret;
}
static int ispkcs11(const dst_key_t *key)
{
  RSA *rsa = key->keydata.generic;
  if(rsa->d && *(long *)rsa->d == PKCS11_MAGIC) return 1;
  return 0;
}
static void pkcs11_free(void *cb)
{
  pkcs11_cb *p11;
  if(fnclist == NULL) return;
  p11 = (pkcs11_cb *)cb;
  pkcs11_logout(p11);
  free(p11);
}
static int pkcs11_login(pkcs11_cb *p11,char *keystr)
{
  int rv;
  CK_ULONG               ulNumberOfSlots=MAX_SLOTS;
  CK_SLOT_ID             SlotList[MAX_SLOTS];
  CK_SESSION_HANDLE      hSessionHandle;

  if(p11->logged_in) return 0; /* already logged in */

  if(pkcs11_initlib(NULL)) return -1;

  if((rv=fnclist->C_GetSlotList(TRUE,SlotList,&ulNumberOfSlots)) != CKR_OK) {
    fprintf(stderr,"pkcs11: error: C_GetSlotList %s\n",pkcs11_ret_str(rv));
    return -1;
  }

  /*fprintf(stderr,"pkcs11: %s slot:%d\n",__func__,p11->slot); FOOP */

  /*printf("pkcs11: Found %d slots\n",ulNumberOfSlots);*/
  if((rv=fnclist->C_OpenSession(SlotList[p11->slot],CKF_RW_SESSION | CKF_SERIAL_SESSION,NULL,NULL,&hSessionHandle)) != CKR_OK) {
    fprintf(stderr,"pkcs11: error: C_OpenSession %s\n",pkcs11_ret_str(rv));
    return -1;
  } else {
    /*fprintf(stderr,"pkcs11: C_OpenSession successfull Slot = %d\n",k);*/
  }

  /*fprintf(stderr,"pkcs11: %s PIN:%s\n",__func__,p11->pin); FOOP */

  if(p11->pin) { /* login */
    if(strlen(p11->pin) == 0) {
      char buf[80];
      int i;
      char *p;

      if((p=getenv("PKCS11_LIBRARY_PIN"))) {
	fprintf(stderr,"C_Login\n");/* FOOP */
        if((rv=fnclist->C_Login(hSessionHandle,CKU_USER,(unsigned char *)p,strlen(p))) != CKR_OK 
	   && rv != CKR_USER_ALREADY_LOGGED_IN) {
          fprintf(stderr,"pkcs11: error: C_Login %s\n",pkcs11_ret_str(rv));
          fnclist->C_CloseSession(hSessionHandle);
          return -1;
	}
	p11->sh = hSessionHandle;
	p11->logged_in = 1;
	/*printf("pkcs11: login slot=%d\n",p11->slot);*/
	return 0;
      }

      while(1) {
	if(keystr) fprintf(stderr,"Please enter PIN for %s token: ",keystr); 
	else fprintf(stderr,"Please enter PIN: ");
	if(fgets(buf,sizeof(buf),stdin)) {
	  buf[sizeof(buf) - 1] = '\0';
	  i = strlen(buf) - 1;
	  buf[i] = '\0';
	} else {
	  printf("PIN input error\n");
	  fnclist->C_CloseSession(hSessionHandle);
	  return -1;
	}
	if((rv=fnclist->C_Login(hSessionHandle,CKU_USER,(unsigned char *)buf,i)) != CKR_OK
	   && rv != CKR_USER_ALREADY_LOGGED_IN) {
	  fprintf(stderr,"pkcs11: error: C_Login %s\n",pkcs11_ret_str(rv));
	} else {
#ifdef PKCS11_SINGLE_SESSION /* security problem */
	  free(p11->pin);
	  p11->pin = strdup(buf); /* remember it */
#endif
	  break;
	}
      }
    } else {
      if((rv = fnclist->C_Login(hSessionHandle,CKU_USER,(unsigned char *)p11->pin,strlen(p11->pin))) != CKR_OK
	 && rv != CKR_USER_ALREADY_LOGGED_IN) {
	fprintf(stderr,"pkcs11: error: C_Login %s\n",pkcs11_ret_str(rv));
	fnclist->C_CloseSession(hSessionHandle); 
	return -1;
      }
    }
  }
  p11->sh = hSessionHandle;
  p11->logged_in = 1;
  /*printf("pkcs11: login slot=%d\n",p11->slot);*/

  return 0;
}
static int pkcs11_logout(pkcs11_cb *p11)
{
  int rv;
  CK_SESSION_HANDLE sh;

  if(p11->magic != PKCS11_MAGIC) {
    fprintf(stderr,"pkcs11: error: invalid control block\n");
    return -1;
  }
  if(p11->logged_in == 0) return 0;

  sh = p11->sh;
  /*if(sh == 0) return 0; /* not open */

  if(p11->pin) {
    if((rv=fnclist->C_Logout(sh)) != CKR_OK) {
      fprintf(stderr,"pkcs11: error: C_Logout %s\n",pkcs11_ret_str(rv));
    }
  }
  if((rv=fnclist->C_CloseSession(sh)) != CKR_OK) {
    fprintf(stderr,"pkcs11: error: %s C_CloseSession %s\n",__func__,pkcs11_ret_str(rv));
  }

  p11->sh = 0;
  p11->logged_in = 0;
  /*printf("pkcs11: logout slot=%d\n",p11->slot);*/

  fprintf(stderr,"pkcs11: C_Finalize\n"); /* FOOP */
  fnclist->C_Finalize(0);  /* FOOP */
  fnclist = NULL;

  return 0;
}
static int pkcs11_RSA_sign(dst_key_t *key,int type,unsigned char *message,unsigned int messagelen,unsigned char *sign,unsigned int *slen)
{
  int rv;
  CK_MECHANISM smech;
  CK_SESSION_HANDLE  sh;
  pkcs11_cb *p11;
  int ret;

  ret = -1;

  if(key->keydata.generic == NULL) return ret;
  if(((RSA *)key->keydata.generic)->d == NULL) return ret;
  p11 = (pkcs11_cb *)((RSA *)key->keydata.generic)->d;

#ifdef PKCS11_SINGLE_SESSION
  if(pkcs11_login(p11,NULL)) return ret;
#endif

  sh = p11->sh;
  /*printf("pkcs11: signing using %s/%s\n",p11->label,p11->id);*/

  /*
   * Taken directly from OPENSSL source so that their RSA_verify will
   * accept our pkcs11 C_Sign output.
   */
  {
    X509_SIG xsig;
    ASN1_TYPE parameter;
    int i;
    uint8 *p,tmps[1024]; /* good to 8192 bit RSA */
    /*const unsigned char *s = NULL;*/
    unsigned char *s;
    X509_ALGOR algor;
    ASN1_OCTET_STRING digest;

    s = NULL;
    xsig.algor = &algor;
    xsig.algor->algorithm = OBJ_nid2obj(type); /*(NID_sha1);*/
    if(xsig.algor->algorithm == NULL) {
      fprintf(stderr,"pkcs11: error: RSA_F_RSA_SIGN,RSA_R_UNKNOWN_ALGORITHM_TYPE\n");
      goto err;
    }
    if(xsig.algor->algorithm->length == 0) {
      fprintf(stderr,"pkcs11: error: RSA_F_RSA_SIGN,RSA_R_THE_ASN1_OBJECT_IDENTIFIER_IS_NOT_KNOWN_FOR_THIS_MD\n");
      goto err;
    }
    parameter.type = V_ASN1_NULL;
    parameter.value.ptr = NULL;
    xsig.algor->parameter= &parameter;
    xsig.digest = &digest;
    xsig.digest->data = (unsigned char *)message;
    xsig.digest->length = messagelen;
    i = i2d_X509_SIG(&xsig,NULL);
    if(i > (int)(sizeof(tmps) - RSA_PKCS1_PADDING_SIZE)) {
      fprintf(stderr,"pkcs11: error: RSA_F_RSA_SIGN,RSA_R_DIGEST_TOO_BIG_FOR_RSA_KEY\n");
      goto err;
    }
    p = tmps;
    i2d_X509_SIG(&xsig,&p);
    s = tmps;

    smech.mechanism = CKM_RSA_PKCS;
    smech.pParameter = NULL_PTR;
    smech.ulParameterLen = 0;
    if((rv=fnclist->C_SignInit(sh,&smech,p11->priv)) != CKR_OK) {
      fprintf(stderr,"pkcs11: error: C_SignInit %s\n",pkcs11_ret_str(rv));
      goto err;
    }
    if((rv=fnclist->C_Sign(sh,(CK_BYTE_PTR)s,(CK_ULONG)i,(CK_BYTE_PTR)sign,(CK_ULONG *)slen)) != CKR_OK) {
      fprintf(stderr,"pkcs11: error: C_Sign %s\n",pkcs11_ret_str(rv));
      goto err;
    }
    /*printf("Signed: %d byte signature for %d byte msg\n",*slen,i);*/
  }

  ret = 0;
 err:
#ifdef PKCS11_SINGLE_SESSION
  pkcs11_logout(p11);
#endif
  return ret;
}

int pkcs11_initlib(char *pin)
{
  void *pkcs11_hLib=NULL;
  CK_C_GetFunctionList   pGFL  = 0;
  int rv;

  /* remember command line specificed PIN number location */
  if(pkcs11_pin == NULL && pin) {
    pkcs11_pin = pin; 
  }

  if(fnclist) return 0;  /* already open and associated */

  fprintf(stderr,"pkcs11: %s\n",__func__); /* FOOP */

  if(pkcs11_library == NULL || strlen(pkcs11_library) <= 0) {
    /*
     * if defined, then use pkcs11
     *
     * possible locations:
     * "/home/dnssec/AEP/pkcs11.so.3.10",
     * "/usr/lib/opensc-pkcs11.so",
     * "/lib/opensc-pkcs11.so",
     * "/usr/local/lib/opensc-pkcs11.so",
     */
    if((pkcs11_library=getenv("PKCS11_LIBRARY_PATH")) == NULL
       || strlen(pkcs11_library) <= 0) {
      /*fprintf(stderr,"You must set PKCS11_LIBRARY_PATH, e.g.,\n \"export PKCS11_LIBRARY_PATH=/usr/lib/opensc-pkcs11.so\"\n");*/
      use_pkcs11 = 0;
      return -1;
    } else {
      use_pkcs11 = 1;
    }
  }
  pkcs11_hLib = dlopen(pkcs11_library,RTLD_LAZY);
  if(!pkcs11_hLib) {
    fprintf(stderr,"pkcs11: error: Failed to open PKCS11 library %s\n",pkcs11_library);
    return -1;
  }
  if((pGFL=(CK_C_GetFunctionList)dlsym(pkcs11_hLib,"C_GetFunctionList")) == NULL) {
    fprintf(stderr,"pkcs11: error: Cannot find GetFunctionList()\n");
    dlclose(pkcs11_hLib);
    return -1;
  }
  if((rv=pGFL(&fnclist)) != CKR_OK) {
    fprintf(stderr,"pkcs11: error: C_GetFunctionList %s\n",pkcs11_ret_str(rv));
    dlclose(pkcs11_hLib);
    fnclist = NULL;
    return -1;
  }
  /*
   * Note: Since we dont know what a vendor is going to do, this might
   * clobber signal handling and other process oriented stuff. So execute
   * this at the begining of main() before we set our own handlers.
   */
  if((rv=fnclist->C_Initialize(NULL)) != CKR_OK) {
    fprintf(stderr,"pkcs11: error: C_Initialize %s\n",pkcs11_ret_str(rv));
    dlclose(pkcs11_hLib);
    fnclist = NULL;
    return -1;
  }
  return 0;
}

static int hex2i(char c)
{
  if(c >= '0' && c <= '9') return (int)(c - '0');
  if(c >= 'A' && c <= 'F') return (int)((c - 'A') + 10);
  return -1;
}

#endif /* R_PKCS11 */
